using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{

		#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/16/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Data access layer component implementing database access functionality 
	///	for ordered unit data
	///</summary>

		#endregion

	public class OrderedUnit
	{
		
		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/21/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7350"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7351"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetCurrentCrossmatchData
		/// </summary>
		/// <param name="orderedUnitGuid"></param>
		/// <returns></returns>
		public static DataRow GetCurrentCrossmatchData(Guid orderedUnitGuid)
		{
			if (orderedUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered unit unique identifier").ResString);
			}

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetCurrentCrossmatchResults.orderedunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetCurrentCrossmatchResults.divisioncode, System.Data.SqlDbType.VarChar),
			};

			prms[0].Value = orderedUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
	
			DataTable dt = Common.StoredProcedure.GetDataTable(STOREDPROC.GetCurrentCrossmatchResults.StoredProcName, prms); 

			if (dt.Rows.Count == 0)
			{
				return null;
			}
			else
			{
				return dt.Rows[0];
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1631"> 
		///		<ExpectedInput>OrderedUnitGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable containing ordered unit information.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1632"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets Ordered unit by it's GUID
		/// </summary>
		/// <param name="orderedUnitGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetOrderedUnitByGuid(System.Guid orderedUnitGuid, string divisionCode)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitByGuid.orderedunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnitByGuid.divisioncode, System.Data.SqlDbType.VarChar),
			};

			prms[0].Value = orderedUnitGuid;
			prms[1].Value = divisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitByGuid.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1633"> 
		///		<ExpectedInput>BloodUnitGuid, PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable containing ordered unit information.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1634"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets Ordered unit by it's GUID
		/// </summary>
		/// <param name="bloodUnitGuidList"></param>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		/// CR2899: changing input parameter
		public static DataTable GetOrderedUnitByPatientAndUnit(string bloodUnitGuidList, System.Guid patientGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitByPatientAndUnit.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnitByPatientAndUnit.bloodunitguidlist, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetOrderedUnitByPatientAndUnit.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = bloodUnitGuidList;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitByPatientAndUnit.StoredProcName, prms);

			return ds.Tables[0];
		}
		
		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3748"> 
		///		<ExpectedInput>Valid BloodUnitGuid and division code</ExpectedInput>
		///		<ExpectedOutput>DataTable containing historical ordered unit information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3749"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		///  Gets the historical order information for a blood unit .
		///</summary>
		///<param name="bloodUnitGuid">Blood unit GUID</param>
		///<param name="divisionCode">Division Code</param>
		///<returns>DataTable containing ordered blood unit details</returns>
		public static DataTable GetHistoricalOrderedUnits(System.Guid bloodUnitGuid, string divisionCode)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetHistoricalOrderedUnits.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetHistoricalOrderedUnits.divisioncode, System.Data.SqlDbType.VarChar),
			};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = divisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetHistoricalOrderedUnits.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1635"> 
		///		<ExpectedInput>OrderedComponentGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable containing ordered unit information.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1636"> 
		///		<ExpectedInput>Inavlid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetOrderedUnitByComponent
		/// </summary>
		/// <param name="orderedComponentGuid">orderedComponentGUID</param>
		/// <param name="divisionCode">divisionCode</param>
		/// <returns>Data table</returns>
		public static DataTable GetOrderedUnitByComponent(System.Guid orderedComponentGuid, string divisionCode)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitByComponent.orderedcomponentguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnitByComponent.divisioncode, System.Data.SqlDbType.VarChar)
			};

			prms[0].Value = orderedComponentGuid;
			prms[1].Value = divisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitByComponent.StoredProcName, prms);

			return ds.Tables[0];
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1637"> 
		///		<ExpectedInput>OrderedComponentGuid, DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable containing ordered unit count.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1638"> 
		///		<ExpectedInput>Inavlid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty Datatable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetOrderedUnitCount
		/// </summary>
		/// <param name="orderedComponentGuid">orderedComponentGUID</param>
		/// <param name="divisionCode">divisionCode</param>
		/// <returns>Data table</returns>
		public static DataTable GetOrderedUnitCount(System.Guid orderedComponentGuid, string divisionCode)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitCount.orderedcomponentguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnitCount.divisioncode, System.Data.SqlDbType.VarChar)
			};

			prms[0].Value = orderedComponentGuid;
			prms[1].Value = divisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitCount.StoredProcName, prms);

			return ds.Tables[0];
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1639"> 
		///		<ExpectedInput>PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable containing ordered units information.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1645"> 
		///		<ExpectedInput>Invalid GUid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetOrderedUnitsForPatient
		/// </summary>
		/// <param name="patientGuid">patientGUID</param>
		/// <returns>Data Table</returns>
		public static DataTable GetOrderedUnitsForPatient(System.Guid patientGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitsForPatient.patientguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnitsForPatient.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitsForPatient.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2135"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable containing ordered units information.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2136"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetOrderedUnitsForPatient
		/// </summary>
		/// <returns>Data Table</returns>
		public static DataTable GetOrderedUnitsForUnit()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitsForUnit.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitsForUnit.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4113"> 
		///		<ExpectedInput>PatientSpecimenGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of OrderedUnits</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4114"> 
		///		<ExpectedInput>Empty PatientSpecimenGuid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public static function returns data table of blood units currently assigned to a given specimen
		/// </summary>
		/// <param name="patientSpecimenGuid">Patient specimen unique identifier</param>
		/// <returns>Data table of blood unit data</returns>
		public static DataTable GetOrderedUnitsForSpecimen(System.Guid patientSpecimenGuid)
		{
			if (patientSpecimenGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient specimen unique identifier").ResString);
			}
		
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitsForPatientSpecimen.patientspecimenguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnitsForPatientSpecimen.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = patientSpecimenGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode; 

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitsForPatientSpecimen.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3667"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of OrderedUnits</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3668"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets all units in the division that are assigned or crossmatched to patients
		/// </summary>
		/// <returns>Data Table</returns>
		public static DataTable GetOrderedUnits()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitsForDivision.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitsForDivision.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="1646"> 
		///		<ExpectedInput>BloodunitGuid, DivisionCode, PatientGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1650"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks if the unit is already crossmatched for the same patient
		/// </summary>
		/// <param name="bloodUnitGuid">bloodUnitGuid</param>
		/// <param name="divisionCode">divisionCode</param>
		/// <param name="patientGuid">patientGUID</param>
		/// <returns></returns>
		public static DataTable CheckIsAlreadyCrossMatched(System.Guid bloodUnitGuid, string divisionCode, System.Guid patientGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.CheckIsAlreadyCrossMatched.bloodunitguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.CheckIsAlreadyCrossMatched.divisioncode, SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.CheckIsAlreadyCrossMatched.patientguid, SqlDbType.UniqueIdentifier)
			};
 
			prms[0].Value = bloodUnitGuid;
			prms[1].Value = divisionCode;
			prms[2].Value = patientGuid;
			
			return Common.StoredProcedure.GetData(STOREDPROC.CheckIsAlreadyCrossMatched.StoredProcName, prms).Tables[0];
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2941"> 
		///		<ExpectedInput>BloodUnitGuid, Division</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2942"> 
		///		<ExpectedInput>Inavlid bloodUnitguid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a data table of ordered units
		/// </summary>
		///<param name="bloodUnitGuid">Blood unit GUID</param>
		///<param name="divisionCode">divisionCode</param>
		public static DataTable GetOrderedUnit(System.Guid bloodUnitGuid, string divisionCode)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnit.bloodunitguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnit.divisioncode, SqlDbType.VarChar)
			};
 
			prms[0].Value = bloodUnitGuid;
			prms[1].Value = divisionCode;
            
			return Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnit.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3028"> 
		///		<ExpectedInput>Valid Ordered Test GUID or valid Ordered Component GUID</ExpectedInput>
		///		<ExpectedOutput>Valid datatable of all ordered units for repeat XM</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3029"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the list of all Ordered Units that are crossmatched based on 
		/// OrderedTest GUID or OrderComponent GUID.
		/// </summary>
		/// <param name="orderedTestGuid">Ordered Test GUID</param>
		/// <param name="orderedComponentGuid">Ordered Component GUID</param>
		/// <returns>DataTable</returns>
		public static DataTable GetOrderedUnitsForRepeatXM(Guid orderedTestGuid,Guid orderedComponentGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetOrderedUnitsForRepeatXM.orderedtestguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnitsForRepeatXM.orderedcomponentguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetOrderedUnitsForRepeatXM.divisioncode, System.Data.SqlDbType.Char)
			};

			if (orderedTestGuid == Guid.Empty)
			{
				prms[0].Value = null;
			}
			else
			{
				prms[0].Value = orderedTestGuid;
			}

			if (orderedComponentGuid == Guid.Empty)
			{
				prms[1].Value = null;
			}
			else
			{
				prms[1].Value = orderedComponentGuid;
			}
			
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetOrderedUnitsForRepeatXM.StoredProcName, prms);
			
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3022"> 
		///		<ExpectedInput>PatientGuid, BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3023"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// This returns a boolean indicating if the patient has already had an
		/// incompatible crossmatch with this unit.
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static bool IncompatibleCrossmatchResultsExist(System.Guid patientGuid, System.Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.IncompatibleCrossmatchResultsExist.PatientGuid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.IncompatibleCrossmatchResultsExist.BloodUnitGuid, SqlDbType.UniqueIdentifier)
			};
 
			prms[0].Value = patientGuid;
			prms[1].Value = bloodUnitGuid;
            
			return ( new Common.StoredProcedure().TransactionalGetValue(STOREDPROC.IncompatibleCrossmatchResultsExist.StoredProcName,prms) == 1 );
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1653"> 
		///		<ExpectedInput>Valid ordered units tables</ExpectedInput>
		///		<ExpectedOutput>Success indicator set to true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1654"> 
		///		<ExpectedInput>Inavalid Comment/ lastUpdate function id</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Release one or more ordered units from patient assignment.
		/// This version does not update the Unit Status. 
		/// </summary>
		/// <param name="orderedUnit">Data table of ordered units to be released</param>
		/// <param name="workloadEvent">Datatable of workload event data</param>
		/// <param name="releaseComment">String containing release comment to apply to all ordered units</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns>Success indicator</returns>
		public static bool ReleaseOrderedUnits(DataTable orderedUnit, DataTable workloadEvent, string releaseComment, Common.UpdateFunction lastUpdateFunctionId, DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			Common.Utility.RequireNonNullColumns(orderedUnit, STOREDPROC.ReleaseOrderedUnits.RequiredParameters);
			//
			if ((releaseComment == null)||(releaseComment.Length==0)||(releaseComment.Length>510))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.FieldLengthError("release comment","1","510").ResString);
			}
			//
			if (lastUpdateFunctionId == Common.UpdateFunction.Invalid)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("last update function id").ResString);
			}
			//
			DataTable dtOrderedUnitComment = new DataTable(TABLES.OrderedUnitComment.TableName);
			dtOrderedUnitComment.Columns.Add(TABLES.OrderedUnitComment.OrderedUnitCommentGuid, typeof(System.Guid));
			dtOrderedUnitComment.Columns.Add(TABLES.OrderedUnitComment.OrderedUnitGuid);
			dtOrderedUnitComment.Columns.Add(TABLES.OrderedUnitComment.OrderedUnitCommentText);
			dtOrderedUnitComment.Columns.Add(TABLES.OrderedUnitComment.LastUpdateUser);
			foreach(DataRow drUnit in orderedUnit.Rows)
			{
				DataRow drComment = dtOrderedUnitComment.NewRow();
				drComment[TABLES.OrderedUnitComment.OrderedUnitCommentGuid] = System.Guid.NewGuid();
				drComment[TABLES.OrderedUnitComment.OrderedUnitGuid] = drUnit[TABLES.OrderedUnit.OrderedUnitGuid];
				drComment[TABLES.OrderedUnitComment.OrderedUnitCommentText] = releaseComment;
				drComment[TABLES.OrderedUnitComment.LastUpdateUser] = Common.LogonUser.LogonUserName;
				dtOrderedUnitComment.Rows.Add(drComment);
			}
			//
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.ReleaseOrderedUnits.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(orderedUnit, lastUpdateFunctionId));
			//
			spArray.Add(STOREDPROC.InsertOrderedUnitComment.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedUnitComment, lastUpdateFunctionId));

			spArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(workloadEvent, lastUpdateFunctionId));
			//
			if (dtBCEMessages != null)
			{
				spArray.Add(STOREDPROC.HL7InsertMessageLog.StoredProcName);
				dtArray.Add(dtBCEMessages);
			}
			
			return ((new Common.StoredProcedure()).TransactionalGetValue(spArray, dtArray)==0);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/13/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7247"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7248"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Releases any ordered units for a given blood unit
		/// Implements BR_11.05
		/// </summary>
		/// <param name="dtOrderedUnit">Data table of ordered units to be released</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns>Success indicator</returns>
		public static bool ReleaseOrderedUnits(DataTable dtOrderedUnit, Common.UpdateFunction lastUpdateFunctionId, DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			Common.Utility.RequireNonNullColumns(dtOrderedUnit, STOREDPROC.ReleaseOrderedUnitsByBloodUnit.RequiredParameters);
			//
			if (lastUpdateFunctionId == Common.UpdateFunction.Invalid)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("last update function id").ResString);
			}
			//
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			spArray.Add(STOREDPROC.ReleaseOrderedUnitsByBloodUnit.StoredProcName);
			dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtOrderedUnit, lastUpdateFunctionId));
			//
			if (dtBCEMessages != null)
			{
				spArray.Add(STOREDPROC.HL7InsertMessageLog.StoredProcName);
				dtArray.Add(dtBCEMessages);
			}
			//
			return ( (new Common.StoredProcedure()).TransactionalGetValue( spArray, dtArray ) == 0 );
		}

		/// <summary>
		/// Get Patient based on OrderedUnitGuid
		/// CR 2941
		/// </summary>
		/// <param name="orderedUnitGuid"></param>
		/// <returns></returns>
		public static DataTable GetPatientForOrderedUnit(System.Guid orderedUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPatientForOrderedUnit.orderedunitguid, SqlDbType.UniqueIdentifier)
			};
 
			prms[0].Value = orderedUnitGuid;
            
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetPatientForOrderedUnit.StoredProcName, prms);

			return ds.Tables[0];
		}
	}
}